/*
	BSDSocket.cpp
	This file serves as the wrapper for all sockets in *NIX.
	Copyright 2002 Ben Everett and Chris Horlick.
	All Rights Reserved.
*/

#include "pch.h"

#ifndef _WIN32
// *NIX

// Engine Includes
#include "Socket.h"
#include "BSDSocket.h"

using namespace CPPMUD;

// Constructor & Deconstructor
CBSDSocket::CBSDSocket()
{
}

CBSDSocket::~CBSDSocket()
{
}

// Functions
ErrRet CBSDSocket::CloseClient(int nClientID)
/*
	Closes a client's connection by their client descriptor.
	PRE: Server has been initialized properly, pClient is a valid pointer to a client descriptor.
	POST: Client's connection closed.
*/
{
	std::vector<CClientDescriptor>::iterator itRem = m_vecClients.begin();

	// Remove the client from the client list.
	for (int i = 0; i < m_vecClients.size(); ++i, ++itRem)
	{
		if (nClientID == m_vecClients[i].m_nClientID)
		{
			std::cout << "Closing client #" << m_vecClients[i].m_nClientID << "'s connection." << 
				std::endl;

			// Close the socket.
			close(m_vecClients[i].m_sSocket);

			m_vecClients.erase(itRem);
		}
	}

	return errNone;
}

ErrRet CBSDSocket::CloseSocket(SOCKET sSocket)
/*
	Closes a client's connection by their socket descriptor.
	PRE: Server has been initialized properly, sSocket is a valid socket descriptor.
	POST: Client's connection closed.
*/
{
	std::cout << "Closing socket, connection timed out." << std::endl;

	// Close the socket.
	close(sSocket);

	return errNone;
}

ErrRet CBSDSocket::CloseSockets()
/*
	Closes all client sockets and finally the server socket.
	PRE: Server was initialized properly.
	POST: All sockets closed properly.
*/
{
	// Close all client's connections.
	for (int i = 0; i < m_vecClients.size(); ++i)
		close(m_vecClients[i].m_sSocket);

	// Close the server socket so no-one else can connect.
	close(m_sSocket);

	// Clear the client list.
	m_vecClients.clear();

	return errNone;
}

ErrRet CBSDSocket::Connect(int nPort)
/*
	Connects to a server on the given port.
	PRE: Server socket has been initialized properly, nPort is a valid port.
	POST: Server is connected to a remote server.
*/
{
	// For future use, to allow servers to be inter-connected.

	return errNone;
}

ErrRet CBSDSocket::InitSockets()
/*
	Initializes the server socket.
	PRE: None.
	POST: Server socket initialized.
*/
{
	int nReuseAddr = 1;

	// Obtain the server socket.
	m_sSocket = socket(AF_INET, SOCK_STREAM, 0);
	if (m_sSocket < 0)
		return errSocketFailCreate;

	// Setup non-blocking on the socket
	NonBlocking(m_sSocket);

	return errNone;
}

ErrRet CBSDSocket::Listen(int nPort)
/*
	Listens for client connections on the given port.
	PRE: Server has been initialized properly, nPort is a valid port.
	POST: Server is listening for incoming client connections.
*/
{
	m_saAddress.sin_family = AF_INET;
	m_saAddress.sin_addr.s_addr = htonl(INADDR_ANY);
	m_saAddress.sin_port = htons(nPort);

	// Bind the socket.
	if (bind(m_sSocket, (sockaddr*)&m_saAddress, sizeof(m_saAddress)) < 0)
	{
		close(m_sSocket);

		return errSocketFailCreate;
	}

	// Listen for any incoming connections.
	if (listen(m_sSocket, 10) != 0)
	{
		close(m_sSocket);

		return errSocketFailListen;
	}

	return errNone;
}

ErrRet CBSDSocket::ReadData(char *czData, int nBufferSize, int nClientID)
/*
	Reads data from the client descriptor and copies it into czData.
	PRE: Server has been initialized properly, czData is a valid pointer to the read buffer, 
		pClient is a valid pointer to a client descriptor, nBufferSize is the length of czData..
	POST: Client data has been read.
*/
{
	int nIndex = GetIndexFromClientID(nClientID);
	int nRead = 0;

	if (nIndex < 0)
		return errUnknownClient;
	
	nRead = recv(m_vecClients[nIndex].m_sSocket, czData, nBufferSize, 0);

	if (nRead < 0)
		return errSocketTimeOut;
	else if (nRead == 0)
		return errSocketFailRead;

	return errNone;
}

ErrRet CWinSocket::SendData(int nClientID)
/*
	Sends as much data as possible in nClientID's outgoing buffer.
	PRE: Server has been initialized properly, nClientID is a valid client ID.
	POST: As much data as possible is sent from the client's outgoing buffer.
*/
{
	int nIndex = GetIndexFromClientID(nClientID);
	int nLen = strlen(m_vecClients[nIndex].m_czOutgoingData);
	int nSent = 0;

	// Send as much data as possible to the client.
	nSent = write(m_vecClients[nIndex].m_sSocket, m_vecClients[nIndex].m_czOutgoingData, 
		strlen(m_vecClients[nIndex].m_czOutgoingData), 0);

	// If nSent is < 0 the socket timed out, if it = 0 there was an error somewhere, if > 0 nSent 
	//    removes that many char's from the outgoing buffer.
	if (nSent < 0)
		return errSocketTimeOut;
	else if (nSent == 0)
		return errSocketFailWrite;
	else
	{
		memmove(m_vecClients[nIndex].m_czOutgoingData, m_vecClients[nIndex].m_czOutgoingData + 
			nSent, 2048 - nSent);
		memset(m_vecClients[nIndex].m_czOutgoingData + (nLen - nSent), 0, sizeof(char) * 
			(nLen - nSent));
	}

	return errNone;
}

void CBSDSocket::NonBlocking(SOCKET sSocket)
/*
	Sets the given socket to non-blocking mode.
	PRE: Server has been initialized properly, sSocket is a valid socket descriptor.
	POST: Socket set to non-blocking mode.
*/
{
	int nOptions = fcntl(sSocket, F_GETFL);

	if (nOptions < 0)
		return;

	nOptions |= O_NONBLOCK;
	if (fcntl(sSocket, F_SETFL, nOptions) < 0)
		return;
}

#endif
